# HealthRegen by David Bowland
# ./addons/eventscripts/healthregen/healthregen.py

# >>> To configure this addon please see healthregen.cfg (created when the addon is first loaded) <<<

"""
Provides steady health regeneration for all players up to an optional, maximum amount.
Requires only ES 2.1+
"""


import cfglib
import es
import gamethread
import playerlib

from path import path

import psyco
psyco.full()


info = es.AddonInfo()
info.name     = 'HealthRegen'
info.version  = '7'
info.basename = 'healthregen'
info.url      = 'http://addons.eventscripts.com/addons/view/' + info.basename
info.author   = 'SuperDave'


addonpath = path(es.getAddonPath(info.basename))

###

config = cfglib.AddonCFG(addonpath.joinpath(info.basename + '.cfg'))


config.text(info.name + ' release %s options' % info.version)
config.text('./addons/eventscripts/%(basename)s/%(basename)s.cfg' % {'basename': info.basename})
config.text(info.url)
config.text('')
config.text('Load this addon with: es_load ' + info.basename)
config.text('\n')

config.text('***** General options *****')
cvar_threshold = config.cvar('healthregen_threshold', 75, 'Maximum health that can be achieved with regeneration')
cvar_roundmax  = config.cvar('healthregen_roundmax',  0,  'Maximum amount of health that can be regenerated in a round--set to 0 for no limit')
cvar_team      = config.cvar('healthregen_team',      0,  '0 = all players regenerate, 1 = only Terrorists regenerate, 2 = only Counter-Terrorists regenerate')
config.text('\n')

config.text('***** Regeneration start options *****')
cvar_start_delay  = config.cvar('healthregen_start_delay',  5,  'Number of seconds a player must remain unhurt to start regeneration--set to 0 for continuous regeneration')
cvar_start_health = config.cvar('healthregen_start_health', 40, 'Minimum amount of health players will have after the first regeneration iteration--set to 0 for no minimum')
config.text('\n')

config.text('***** Regeneration iteration options *****')
cvar_iteration_delay  = config.cvar('healthregen_iteration_delay',  1, 'Seconds between regeneration iterations')
cvar_iteration_amount = config.cvar('healthregen_iteration_amount', 2, 'Amount of health to regenerate each iteration')

config.write()

###

class RegenPlayer(object):
   """ Handles player regeneration """

   def __init__(self, userid):
      self.userid = userid
      self.remaining = int(cvar_roundmax) if cvar_roundmax else -1
      self.delayname = 'healthregen_' + str(self.userid)

   def playerHurt(self):
      """ Creates a delay to regenerate health for the player """
      # Clear the existing heal delay for the player
      self.clearDelay()

      # Make sure we should start our heal delay
      current_player = playerlib.getPlayer(self.userid)
      if cvar_threshold <= current_player.getHealth():
         return
      regen_team = int(cvar_team)
      if regen_team in (1, 2) and current_player.getAttribute('teamid') != regen_team + 1:
         return

      # Start the heal delay
      if self.remaining:
         if cvar_start_delay:
            delay = cvar_start_delay
         else:
            delay = cvar_iteration_delay
         gamethread.delayedname(delay, self.delayname, self.regenHealth)

   def regenHealth(self):
      """
      Checks to make sure the player still needs health
      Adds the required health to the player and executes another iteration if the player needs health after addition
      """
      # Get player information
      player = playerlib.getPlayer(self.userid)
      health_now = player.getHealth()
      health_max = int(cvar_threshold)

      # Determine whether we should regenerate health
      if health_max <= health_now or not self.remaining:
         return

      # Determine the amount of health to give the player
      health_min = int(cvar_start_health)
      if health_now < health_min:
         health_addition = health_min - health_now
      else:
         health_addition = int(cvar_iteration_amount)
         if health_now + health_addition > health_max:
            health_addition = health_max - health_now
      if self.remaining >= 0:
         health_addition = min(health_addition, self.remaining)
         self.remaining -= health_addition

      # Set the player's new health
      health_now += health_addition
      player.setHealth(health_now)

      # Restart the loop if necessary
      if self.remaining and health_now < health_max:
         gamethread.delayedname(cvar_iteration_delay, self.delayname, self.regenHealth)

   def clearDelay(self):
      """ Removes the delay corresponding to the player """
      gamethread.cancelDelayed(self.delayname)

###

class RegenManager(dict):
   """ Handles players regenerating health """

   def __getitem__(self, userid):
      """ Returns a RegenPlayer instance for the given userid """
      userid = int(userid)
      if userid not in self:
         self[userid] = RegenPlayer(userid)
      return super(RegenManager, self).__getitem__(userid)

   def __delitem__(self, userid):
      """ Removes a userid from the dictionary of regenerating players """
      userid = int(userid)
      if userid in self:
         self[userid].clearDelay()
         super(RegenManager, self).__delitem__(userid)

   def clear(self):
      """
      Clears the dictionary of regenerating players and
      cancels all regeneration delays
      """
      for userid in self:
         self[userid].clearDelay()
      super(RegenManager, self).clear()

players = RegenManager()

###

def load():
   config.execute()


def es_map_start(event_var):
   players.clear()


def round_start(event_var):
   players.clear()


def player_hurt(event_var):
   players[event_var['userid']].playerHurt()


def player_death(event_var):
   del players[event_var['userid']]


def player_disconnect(event_var):
   del players[event_var['userid']]


def unload():
   players.clear()

   for cvar in config.getCvars():
      es.ServerVar(cvar).set(0)